﻿using System.Collections;
using System.Collections.Generic;
using TMPro;
using UnityEngine;
using UnityEngine.EventSystems;

public class RoundResultMenu : MonoBehaviour
{
    private GameObject _BackgroundPanel;

    private void Awake()
    {
        //Subscribe to events
        InputManager.Instance.InputStateChanged += OnInputStateChanged;
        _BackgroundPanel = gameObject.FindChild("BackgroundPanel");
        _BackgroundPanel.GetComponent<PhaseAnimation>().BounceInCompleted += RoundResultMenu_BounceInCompleted;
        _BackgroundPanel.GetComponent<PhaseAnimation>().BounceOutCompleted += RoundResultMenu_BounceOutCompleted;
    }

    private void RoundResultMenu_BounceInCompleted(object sender, System.EventArgs e)
    {
        if (InputManager.Instance.InputState == Constants.InputState.Controller)
        {
            //We've just changed to a controller so focus on the next round button
            EventSystem.current.SetSelectedGameObject(null);
            EventSystem.current.SetSelectedGameObject(_BackgroundPanel.FindChild("NextRoundButton"));
        }

        //Bouncing is done so enable the UI
        gameObject.Enable();
    }

    private void RoundResultMenu_BounceOutCompleted(object sender, System.EventArgs e)
    {
        //Bouncing is done so hide the round result UI and move onto the next begin phase
        GameController.Instance.RoundResultScreen.SetActive(false);
        GameController.Instance.ChangeGamePhase(new BeginPhase());
    }

    private void OnInputStateChanged(object sender, InputManager.InputStateChangedEventArgs e)
    {
        if (e.NewState == Constants.InputState.Controller && gameObject.activeSelf)
        {
            //We've just changed to a controller so focus on the next round button
            EventSystem.current.SetSelectedGameObject(null);
            EventSystem.current.SetSelectedGameObject(_BackgroundPanel.FindChild("NextRoundButton"));
            e.Handled = true;
        }
    }

    private void OnDestroy()
    {
        //Unsubscribe from events
        InputManager.Instance.InputStateChanged -= OnInputStateChanged;
        _BackgroundPanel.GetComponent<PhaseAnimation>().BounceInCompleted -= RoundResultMenu_BounceInCompleted;
        _BackgroundPanel.GetComponent<PhaseAnimation>().BounceOutCompleted -= RoundResultMenu_BounceOutCompleted;
    }

    public void MoveToThis(Constants.PlayerRole winner, int round)
    {
        //We're moving to the round result menu so disable the UI
        gameObject.Disable();

        TextMeshProUGUI resultText = _BackgroundPanel.FindChild("RoundResultText").GetComponent<TextMeshProUGUI>();

        if (winner == Constants.PlayerRole.Hider)
        {
            //Hider won, so set the text
            resultText.SetText("HIDER WINS ROUND " + round + "! SWITCHING SIDES...");
        }

        else
        {
            //Seeker won, so set the text
            resultText.SetText("SEEKER WINS ROUND " + round + "! SWITCHING SIDES...");
        }

        //Begin animating the panel
        _BackgroundPanel.gameObject.transform.localScale = new Vector3(0.0f, 0.0f, 1.0f);
        _BackgroundPanel.GetComponent<Animator>().Play("BounceIn", -1, 0.0f);
    }

    public void MoveToNext()
    {
        //We're moving away so disable the UI and begin animating the panel
        gameObject.Disable();
        _BackgroundPanel.GetComponent<Animator>().Play("BounceOut", -1, 0.0f);
    }
}
